<?php

/**
 * Process Form
 * Sanitize and store form input, then email it 
 *
 * @package Domain Broker 2
 * @author Derek Loewen <derek@loewenweb.com>
 */

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;

// Run processing if the form was submitted and the Visual Editor is not active

if( 
  $_SERVER['REQUEST_METHOD'] == 'POST'
  && isset($_GET['editor']) === false
) {

  // General variables
  $domains = lowercaseArrayKeys($domains); // Convert domain names to lower case, avoids array reference issues
  $error = '';
  $error_code = '';
  $success = false;
  $success_message = '<p>Your offer has been sent, thanks!</p>';
  $honeypot_field = 'url';
  $current_domain = punycodeToUtf8($_SERVER['HTTP_HOST']); 
  $missing_fields = [];
  $field_data = [];

  // Store form fields
  $fields = [
    'purchase_type' => 'Purchase Type',
    'offer' => 'Offer',
    'name'  => 'Full Name',
    'email' => 'Email'
  ];
  $fields_not_required = ['purchase_type'];
  // If the Domain Select Field is enabled, add it
  if( $config['enable_domain_select_field'] === true ) {
    $fields['domain'] = 'Domain';
  }
  // If the Message Field is enabled, add it, and make it optional
  if( $config['enable_message_field'] === true ) {
    $fields['message'] = 'Message';
    $fields_not_required['message'] = '';
  }
  // Validate, sanitize, and store contact form input
  $result = validateAndSanitize($fields, $fields_not_required);
  $field_data = $result['field_data'];
  $missing_fields = $result['missing_fields'];

  // Prepare purchase type variable
  if( isset($field_data['purchase_type'])) {
    $purchase_type = $field_data['purchase_type'];
  }
  if( !isset($field_data['purchase_type']) || $field_data['purchase_type'] == '' ) {
    $purchase_type = 'buy'; // Set default value to "buy" if blank
  }
  // The selected domain defaults to the current website address
  // It changes if the user has submitted a different domain from the 
  // domain select field in the offer form.
  $selected_domain = $current_domain;
  if( isset($_POST['domain']) ) {
    $selected_domain = strtolower( filter_var( $_POST['domain'], FILTER_SANITIZE_STRING ) );
  }
  // Prepare current domain price and lease price variables
  $price = $domains[$selected_domain]['price'];
  $lease_price = '';
  if( isset($domains[$selected_domain]['lease_price']) ) {
    $lease_price = $domains[$selected_domain]['lease_price'];
  }

  // Check if honeypot field is filled out
  // if it is, stop here and give a generic error
  if( isset( $_POST[$honeypot_field]) && $_POST[$honeypot_field] != '' ) {
    $error .= '<li>There was an error processing the form. Please refresh the page and try again.</li>';
  }

  // Reject offers that are lower than a percentage of the domain's price
  // example: if the domain price is 1000, and the reject_price_percentage 
  // config setting is 80, offers below 800 will be rejected
  if( 
    intval( $config['reject_price_percentage'] ) > 0 
    && isset( $domains[$selected_domain] )
    && ( $price != '' || $lease_price != '' )
  ) {

    // Sanitize and store variables
    
    $percentage = intval( $config['reject_price_percentage'] );
    // Normalize the offer and price into a regular integer
    // Supports strings like "$1,000", "1000", "1k"
    // offer
    $offer = trim($_POST['offer']);
    $multiplier = ( strtolower(substr($offer, -1)) == 'k' ) ? 1000 : 1;
    $offer = floatval( preg_replace("/[^0-9\.]/", "", $offer) ) * $multiplier;
    // price
    $multiplier = ( strtolower(substr($price, -1)) == 'k' ) ? 1000 : 1;
    $price = floatval( preg_replace("/[^0-9\.]/", "", $price) ) * $multiplier;
    // lease price
    $multiplier = ( strtolower(substr($lease_price, -1)) == 'k' ) ? 1000 : 1;
    $lease_price = floatval( preg_replace("/[^0-9\.]/", "", $lease_price) ) * $multiplier;
    
    // Calculate minimum offer price

    if( $purchase_type === 'buy' ) {
      $min_offer = round( $price * ($percentage/100) );
    } else {
      $min_offer = round( $lease_price * ($percentage/100) );
    }
    
    // Error if offer is not greater than minimum offer

    if( $offer < $min_offer) {
      $error = '<li>Please try a higher offer.</li>';
    }

  }

  if( $config['enable_recaptcha'] === true ) {

    // Verify reCAPTCHA response

    if( 
      isset($_POST['g-recaptcha-response']) 
      && $_POST['g-recaptcha-response'] != ''
    ) {

      // Set the URL to post to and fields to send
      $url = 'https://www.google.com/recaptcha/api/siteverify';
      $recaptcha_response = filter_var( $_POST['g-recaptcha-response'], FILTER_SANITIZE_STRING );
      $recaptcha_fields = [
        'secret' => $config['recaptcha_private_key'],
        'response' => $recaptcha_response,
        'remoteip' => $_SERVER['REMOTE_ADDR']
      ];

      // Convert fields to URL string
      $fields_string = '';
      foreach( $recaptcha_fields as $key => $value ) { 
        $fields_string .= $key.'='.$value.'&'; 
      }
      rtrim( $fields_string, '&' );

      // Open connection
      $ch = curl_init();

      // Set the return, url, number of post vars, post data
      curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
      curl_setopt( $ch, CURLOPT_URL, $url );
      curl_setopt( $ch, CURLOPT_POST, count($recaptcha_fields) );
      curl_setopt( $ch, CURLOPT_POSTFIELDS, $fields_string );

      // Execute post and decode result
      $result = json_decode(curl_exec($ch), true);

      // Close connection
      curl_close($ch);

      // Validate hostname - it has to be one of the domains set up inside DB2
      $result['hostname'] = punycodeToUtf8($result['hostname']);
      if( ! isset( $domains[$result['hostname']] ) ) {
        $error = '<li>This domain name is not valid.</li>';
      }

      // Verify recaptcha passed validation
      if( ! $result['success'] ) {
        $error = '<li>The reCAPTCHA wasn\'t right, please try again.</li>';
      }
      
    } else {
      $error = '<li>reCAPTCHA must be filled out.</li>';
    }

  }

  // Set up missing fields error message if there are missing fields present

  foreach ($missing_fields as $field) {
    $error .= "<li>{$fields[$field]} must be filled out</li>";
  }

  // Send email if there are no errors 

  if( $error == '' && empty($missing_fields) ) {
    
    $mail = new PHPMailer;

    // If SMTP server and email password are filled out, use SMTP
    if( 
      $config['smtp_server'] != ''
      && $config['recipient_email_password'] != ''
     ) {
      // Verbose debugging messages
      //$mail->SMTPDebug = 3;  
      // Set mailer to use SMTP
      $mail->isSMTP();                                      
      // Specify main SMTP server
      $mail->Host = $config['smtp_server'];  
      // Enable SMTP authentication
      $mail->SMTPAuth = true;                               
      // SMTP username
      $mail->Username = $config['recipient_email'];
      // SMTP password                 
      $mail->Password = $config['recipient_email_password'];
      // Enable TLS encryption, 'ssl' also accepted                    
      $mail->SMTPSecure = 'tls';                            
      // TCP port to connect to
      $mail->Port = 587;                        
    }
                       
    // Improve international character support using UTF-8
    // Base64 has been reported to help with this as well
    $mail->Encoding = "base64";
    $mail->CharSet = 'UTF-8';           

    $mail->From = $config['recipient_email'];
    $mail->FromName = $config['recipient_name'];
    // Add a recipient
    $mail->addAddress($config['recipient_email'], $config['recipient_name']);    
    $mail->addReplyTo($field_data['email'], $field_data['name']);

    // Set email format to HTML
    $mail->isHTML(true);

    // Add message field if enabled and filled out
    $message = '';
    if( 
      $config['enable_message_field'] === true
      && trim($field_data['message']) !== ''
    ) {
      $message = 'Message: <br>' . $field_data['message'] . '<br>';
    }

    $mail->Subject = "Domain offer for {$selected_domain}";
    $mail->Body    = "
      {$field_data['name']} &lt;{$field_data['email']}&gt; has submitted an offer for {$selected_domain}!<br>
      Purchase Type: {$purchase_type} <br>
      Offer: {$field_data['offer']} <br>
      $message
      <br>
      -- <br>
      Sent via {$current_domain}<br>
      Submitter IP: <a title='Look up IP location' href='http://ip-api.com/#{$_SERVER['REMOTE_ADDR']}'>{$_SERVER['REMOTE_ADDR']}</a>
    ";
    $mail->AltBody = strip_tags($mail->Body);

    if( !$mail->send() ) {
      $error_code = 'SMTP';
        $error = 'Sorry, the message could not be sent. Please contact me with the info on the contact bar below. <!-- Mailer Error: ' . $mail->ErrorInfo . ' -->';
    } else {
        $success = $success_message;
    }
  }

  // If it's an AJAX request, stop execution here
  if( isset($_GET['ajax']) ) {
    $response = ['error' => $error, 'error_code' => $error_code, 'success' => $success];
    echo json_encode($response); 
    exit;
  }
}