<?php 

/**
 * Functions
 * Theme helpers and utilities
 *
 * @package Domain Broker 2
 * @author Derek Loewen <derek@loewenweb.com>
 */ 

use Algo26\IdnaConvert\IdnaConvert;

/**
 * Convert hexidecimal color to RGB color
 * Source: http://bavotasan.com/2011/convert-hex-color-to-rgb-using-php/
 *
 * @param string $hex hexadecimal color code
 * @return string comma separated R,G,B values
 */
function hex2rgb($hex) {
  $hex = str_replace("#", "", $hex);

  if(strlen($hex) == 3) {
    $r = hexdec(substr($hex,0,1).substr($hex,0,1));
    $g = hexdec(substr($hex,1,1).substr($hex,1,1));
    $b = hexdec(substr($hex,2,1).substr($hex,2,1));
  } else {
    $r = hexdec(substr($hex,0,2));
    $g = hexdec(substr($hex,2,2));
    $b = hexdec(substr($hex,4,2));
  }
  $rgb = [$r, $g, $b];
  return implode(",", $rgb); // returns the rgb values separated by commas
  //return $rgb; // returns an array with the rgb values
}

/**
 * Detect if the current URL is using HTTPS or not
 * Source: https://stackoverflow.com/a/47753742
 * 
 * @return boolean
 */

function isHttps() {
  if( array_key_exists("HTTPS", $_SERVER) && 'on' === $_SERVER["HTTPS"] ) {
    return true;
  }
  if( array_key_exists("SERVER_PORT", $_SERVER) && 443 === (int)$_SERVER["SERVER_PORT"] ) {
    return true;
  }
  if( array_key_exists("HTTP_X_FORWARDED_SSL", $_SERVER) && 'on' === $_SERVER["HTTP_X_FORWARDED_SSL"] ) {
    return true;
  }
  if( array_key_exists("HTTP_X_FORWARDED_PROTO", $_SERVER) && 'https' === $_SERVER["HTTP_X_FORWARDED_PROTO"] ) {
    return true;
  }
  return false;
}

/**
 * Validate and sanitizes POST input, mark missing fields
 * 
 * @param array $fields list of allowed fields, the key being the POSTed field name, 
 *    and the value its human-friendly name
 * @param array $fields_not_required list of fields that can be blank
 * @param array $fields_validation_types associative list of fields with special validation types
 *      example: 'price_tag_caption' => 'html' for allowing HTML characters
 *
 * @return array nested array with 'field_data' and 'missing-fields' 
 *    (sanitized field data and list of fields missing from the POST data, respectively)
 */

function validateAndSanitize($fields, $fields_not_required = [], $fields_validation_types = []) {

  $field_data = [];
  $missing_fields = [];

  foreach( $fields as $key => $value ) {

    // If any field is not present and required,
    // add it to the $missing_fields array
    // ----------------------------------------------------------------
    if( 
      ! isset($fields_not_required[$key])
      && ( ! isset($_POST[$key]) || trim($_POST[$key]) == '' )
    ) {
      
      $missing_fields[] = $key;

    // Field is missing, but not required. Add blank value.
    // ----------------------------------------------------------------
    } elseif ( ! isset($_POST[$key]) ) {

      $field_data[$key] = '';

    // Field is present
    // ----------------------------------------------------------------
    } elseif ( isset($_POST[$key]) ) {

      if( is_array($value) ) { // Handle array - domains 

        // The domain portfolio array is structured like this:
        //
        // 'domainone.com' => [
        //    'nicename' => 'DomainOne.com'
        //    'price' => '$100',
        //    'description' => 'test description',
        //    'not_for_sale' => false,
        //    'image' => ''
        // ),
        // ...
        foreach ($value as $domain) {
          // Sanitize each value
          foreach( $domain as $domain_key => $domain_value) {
            $domain[$domain_key] = filter_var( $domain[$domain_key], FILTER_SANITIZE_STRING );
          }
          $domain_name_lowercase = strtolower( $domain['domain'] );
          // Add new nicename field
          $domain['nicename'] = $domain['domain'];
          // Put sanitized values in $field_data
          $field_data[$key][$domain_name_lowercase] = $domain;
          // Remove extra instance of domain name
          unset($field_data[$key][$domain_name_lowercase]['domain']);
        }

      } else { // Handle strings and other data types

        // If special field type is specified, change sanitization filter to match - only supports HTML for now
        // Else use default string sanitize filter
        if( 
          isset($fields_validation_types[$key])
          && $fields_validation_types[$key] === 'html'
        ) {
          // Remove all tags except specified exceptions, and store string in $field_data
          $allowed_tags = '<div><span><p><a><ul><ol><li><abbr><accronym><address><img><br><q><strong><em><sub><sup>';
          $field_data[$key] = strip_tags( $_POST[$key], $allowed_tags );
        } elseif(
          isset($fields_validation_types[$key])
          && $fields_validation_types[$key] === 'unfiltered'
        ) {
          // Store without sanitization. Only use with fields that must support all HTML, including script tags.
          $field_data[$key] = $_POST[$key];
        } else {
          // Sanitize and store string in $field_data
          $field_data[$key] = filter_var( $_POST[$key], FILTER_SANITIZE_STRING );
        }

        // Validate email with a specific filter
        //
        // Scenarios:
        // 1. Email is required, and invalid - trigger missing
        // 2. Email is not required, not blank, and invalid - trigger missing
        // 3. Email is not required, blank - exit condition
        if(
          $key === 'email' 
          && ( ! isset($fields_not_required[$key]) || $field_data['email'] != '' ) 
        ) {
          if( filter_var( $_POST[$key], FILTER_VALIDATE_EMAIL ) === FALSE ) {
            $missing_fields[] = 'email';
          }
        }
      }
    }
  }

  return array(
    'field_data' => $field_data,
    'missing_fields' => $missing_fields
  );
}

/**
 * Replaces a config placeholder in the format *|SETTINGNAME_VALUE|* 
 * with a specified string
 *
 * Assumes $config array is included beforehand
 *
 * @param string $search string to search for
 * @param string $replace string to replace with
 * @param string $subject to find and replace in
 *
 * @return string text with search/replace operations performed
 */

function replaceConfigPlaceholder($search, $replace, $subject) {
  // Escape single quotes
  $replace_escaped = str_replace("'", "\'", $replace);
  // Replace in string
  $string = str_replace('*'.strtoupper($search . '_VALUE').'*', $replace_escaped, $subject);
  return $string;
}

/**
 * Convert array keys to lowercase
 *
 * @param array $array
 *
 * @return array
 */
function lowercaseArrayKeys($array) {
  return array_change_key_case($array, CASE_LOWER);
}

/**
 * Get current domain and its info
 * 
 * @param array $domains_array array with more domain information
 *
 * @return array with at least 'name' and 'nicename' keys
 */
function getCurrentDomain($domains_array) {
  
  $current_domain = [];
  
  // Get the current domain from the $_SERVER array
  $current_domain['name'] = strtolower( filter_var( $_SERVER['HTTP_HOST'], FILTER_SANITIZE_STRING ) );

  // If the domain is an IDN, convert its punycode to human-readable text
  $current_domain['name'] = punycodeToUtf8($current_domain['name']);
  
  // nicename defaults to the lowercase domain name
  $current_domain['nicename'] = $current_domain['name'];

  // Default image override is nothing
  $current_domain['image'] = '';
  
  // If $domains_array has info for the current domain, overwrite $current_domain
  if( isset($domains_array[$current_domain['name']]) ) {
    $current_domain = array_merge( $current_domain, $domains_array[$current_domain['name']] );
  }

  return $current_domain;
}

/**
 * If the domain is an IDN, convert its punycode to human-readable text
 * 
 * Depends on IdnaConvert class from the algo26-matthias/idna-convert
 * composer package for Internationalized Domain Name conversion
 * 
 * @example "xn--bcher-kva.com" becomes "bücher.com"
 * @example "mydomain.com" does not change (not an IDN)
 * 
 * @param string domain name in puny code, or regular domain name (passes through unaffected)
 * 
 * @return string domain name
 */
function punycodeToUtf8($domain_name) {
  if( substr($domain_name, 0, 4) === 'xn--' ) {
    $converter = new IdnaConvert();
    return $converter->decode($domain_name);
  }

  return $domain_name;
}


/**
 * Generate the <title> element's text
 *
 * @param string $domain_name the domain name to display
 * @param array $domains_array array with more domain information
 *
 * @return string the final title
 */
function getTitle($domain_name, $domains_array) {
  
  // Add domain name
  $title = $domain_name;

  // If the domain is for sale, add "is for sale" text
  if( 
    ! isset($domains_array[$domain_name]['not_for_sale']) 
    || $domains_array[$domain_name]['not_for_sale'] === false 
  ) {
    // Only add the text for domains that are not RTL 
    // since there is a bug with Google Chrome's title 
    // formatting with mixed LTR and RTL strings
    if( !is_rtl($domain_name) ) {
      $title .= ' is for sale!';
    }
  }
  
  return $title;
}

/**
 * Generate a reCAPTCHA token
 * Dependencies: ReCaptchaSecureToken class
 *
 * @param string $public_key
 * @param string $private_key
 *
 * @return string
 */
function generateReCaptchaToken($public_key, $private_key) {
  $recaptchaToken = new \ReCaptchaSecureToken\ReCaptchaToken( ['site_key' => $public_key, 'site_secret' => $private_key] );
  $sessionId = uniqid('recaptcha');
  return $recaptchaToken->secureToken($sessionId);
}

/**
 * Is RTL
 * Check if there RTL characters (Arabic, Persian, Hebrew)
 * 
 * @author	Khaled Attia <sourcecode@khal3d.com>
 * @param	String	$string
 * @return	bool
 */
function is_rtl( $string ) {
	$rtl_chars_pattern = '/[\x{0590}-\x{05ff}\x{0600}-\x{06ff}]/u';
	return preg_match($rtl_chars_pattern, $string);
}
