<?php 

/**
 * Editor
 * Customize your install visually!
 *
 * @package Domain Broker 2
 * @author Derek Loewen <derek@loewenweb.com>
 */ 

// Requre editor parameter in the request
if( isset($_GET['editor']) ) {

  // -------------------------------------------------------------------
  // 1. Check Authentication
  // -------------------------------------------------------------------

  session_start();

  // If authentication data isn't stored, redirect to auth page
  if( $config['editor_login_email'] == '' || $config['editor_login_email'] == 'email@somedomain.com' ) {
    echo '<p><strong>Error:</strong> you must enter your email using the <code style="background:#FFFFCA">editor_login_email</code> setting found in the config.php file. For more details, please refer to the documentation.</p>'; 
    exit;
  } elseif( ! isset($_SESSION['envato_account_email']) ) {
    header("Location:./auth/");
    exit;
  } else {
    // validate authenticated email matches email set in config.php
    if( $_SESSION['envato_account_email'] != $config['editor_login_email'] ) {
      // Does not match
      echo '<p style="font-family: sans-serif;"><strong>Error:</strong> the account you logged in with (' . $_SESSION['envato_account_email'] . ') has a different email than the <code style="background:#FFFFCA">editor_login_email</code> setting found in the config.php file. Please change that setting to the email associated with your account.</p><p style="font-family: sans-serif;"><a href="./auth/">Take me back to the login screen</a></p>'; 
      exit;
    }
  }

  // -------------------------------------------------------------------
  // 2. Process saved changes
  // -------------------------------------------------------------------
  // Changes will be sent using an AJAX request from the editor.js file

  if( isset($_GET['ajax']) ) {

    $domains = lowercaseArrayKeys($domains); // Convert domain names to lower case, avoids array reference issues
    $error = '';
    $success = false;
    $field_data = [];

    $fields_not_required = [
      'domains' => '',
      'enable_domain_portfolio' => '',
      'enable_purchase_type_field' => '',
      'enable_domain_select_field' => '',
      'enable_message_field' => '',
      'ga_tracking_id' => '',
      'enable_recaptcha' => '',
      'recaptcha_public_key' => '',
      'recaptcha_private_key' => '',
      'price_tag_caption' => '',
      'lease_price_tag_caption' => '',
      'image' => '',
      'tint' => '',
      'texture' => '',
      'email' => '',
      'phone' => '',
      'twitter_handle' => '',
      'copyright' => '',
      'head_code' => '',
      'privacy_policy_link' => ''
    ];
    $fields_validation_types = [
      'price_tag_caption' => 'html',
      'lease_price_tag_caption' => 'html',
      'head_code' => 'unfiltered',
      'privacy_policy_link' => 'unfiltered'
    ];

    // Validate, sanitize, and store contact form input

    $result = validateAndSanitize($_POST, $fields_not_required, $fields_validation_types);
    $field_data = $result['field_data'];
    $missing_fields = $result['missing_fields'];

    foreach ($missing_fields as $field) {
      $error .= "<li>" . ucwords($field) . " must be filled out</li>";
    }

    // Save input in config.php file

    if( $error == '' ) {
      
      // Read config template
      if( $config_template = file_get_contents('includes/config-template.php') ) {

        // Go through each config setting
        // If new value for setting was entered, use it
        // example key: 'editor_auth_methods'
        
        // Since 1.4.0 - upgrade from previous versions of config.php
        // Add new settings to config array if not present along with their defaults
        $new_fields = [
          'enable_domain_select_field' => false,
          'enable_purchase_type_field' => false,
          'enable_message_field' => false,
          'copyright' => '',
          'price_tag_caption' => '<abbr title="estimated">est.</abbr> value',
          'lease_price_tag_caption' => 'per month',
          'head_code' => '',
          'privacy_policy_link' => ''
        ];
        foreach( $new_fields as $new_field_key => $new_field_value ) {
          if( ! isset($config[$new_field_key]) ) {
            $config[$new_field_key] = $new_field_value;
          }
        }

        foreach($config as $key => $value) {
          if( isset($field_data[$key]) ) {
            $config[$key] = $field_data[$key];
          }
          // Set specific string for boolean values
          if($config[$key] === TRUE) {
            $config[$key] = 'true';
          }
          if($config[$key] === FALSE) {
            $config[$key] = 'false';
          }
          // Replace placeholder values in template with config data
          if( ! is_array($value) ) {
            // regular value
            $config_template = replaceConfigPlaceholder($key, $config[$key], $config_template);
          } else {
            // array (level 2) ex. 'client_id'
            foreach ($value as $key2 => $value2) {
              $config_template = replaceConfigPlaceholder($key2, $config[$key][$key2], $config_template);
            }
          }
        }

        // Overwrite the domains array found in config.php
        // if form data exists
        if( isset($field_data['domains']) ) {
          // Convert values to lowercase to prevent index mismatch
          $domains = $field_data['domains'];
        } 
        // Build domains array string
        $domains_str = '';
        foreach ($domains as $domain => $arr) {
          if( trim($domain) != '' ) {
            /*
              Array format:

              'domainone.com' => [
                'nicename' => 'DomainOne.com', // preserves case
                'price' => '$100',
                'lease_price' => '$100/mo',
                'enable_purchase_type_field' => false,
                'description' => 'test description',
                'not_for_sale' => false // if true, removes domain from portfolio and hides price/"is for sale" text.
                'image' => 'mountains' // override background image per domain if set
              ],
            */
            
            // Set default for "Not for Sale" setting if none present
            if( 
              ! isset($arr['not_for_sale'])
              || ( isset($arr['not_for_sale']) && $arr['not_for_sale'] == 'false' )
            ) {
              $arr['not_for_sale'] = 'false';
            } else {
              $arr['not_for_sale'] = 'true';
            }

            // Set default for "Lease Price" setting if none present
            if( 
              ! isset($arr['lease_price'])
              || ( isset($arr['lease_price']) && $arr['lease_price'] == '' )
            ) {
              $arr['lease_price'] = '';
            }

            // Set default for "Enable Purchase Type Field" setting if none present
            if( 
              isset($arr['enable_purchase_type_field']) 
              && $arr['enable_purchase_type_field'] == 'false'
            ) {
              $arr['enable_purchase_type_field'] = false;
            } elseif (
              ! isset($arr['enable_purchase_type_field'])
              || $arr['enable_purchase_type_field'] == ''
            ) {
              $arr['enable_purchase_type_field'] = '';
            } else {
              $arr['enable_purchase_type_field'] = true;
            }
                
            $domains_str .= "'$domain' => [\n\t\t'nicename' => '{$arr['nicename']}',\n\t\t'price' => '{$arr['price']}',\n\t\t'lease_price' => '{$arr['lease_price']}',\n\t\t'enable_purchase_type_field' => " . var_export($arr['enable_purchase_type_field'], true) . ",\n\t\t'description' => '{$arr['description']}',\n\t\t'not_for_sale' => {$arr['not_for_sale']},\n\t\t'image' => '{$arr['image']}'\n\t], \n\t";
          }
        }

        // Replace placeholder values for domain portfolio and Domain Broker domains
        $config_template = str_replace('*DOMAINS_VALUE*', $domains_str, $config_template);

        // Remove 'do not edit' notice
        $config_template = str_replace("/* WARNING: DO NOT EDIT THIS FILE DIRECTLY. USE THE WEB-BASED EDITOR FEATURE OR EDIT CONFIG.PHP DIRECTLY */\n", '', $config_template);

        // Write config file
        if( file_put_contents('config.php', $config_template) ) {
          $success = true;
        } else {
          $error = '<li>Config template could not be written to. Aborting save.</li>';
        }
      } else {
        $error = '<li>Config template could not be read. Aborting save.</li>';
      }
    }
    
    // Output response in JSON format then exit
    $response = ['error' => $error, 'success' => $success];
    echo json_encode($response); 
    exit;
  }

  // -------------------------------------------------------------------
  // 3. Set up the Editor interface
  // -------------------------------------------------------------------

  // Find backgrounds and textures
  $images = glob("assets/images/*.{jpg,png}", GLOB_BRACE);
  $backgrounds = [];
  $textures = [];

  foreach ($images as $image) {

    // Remove folder path from image
    $image = strtolower(str_replace('assets/images/', '', $image));

    // Is the image a background (starts with "bg-")?
    if( strpos($image, 'bg-') !== false ) {
      // Remove everything but the name
      $image = str_replace( ['bg-', '.jpg'], '', $image);
      // Add to backgrounds array
      $backgrounds[] = $image;
    } 
    // Is the image a texture (starts with "texture")? 
    elseif( strpos($image, 'texture-') !== false ) {
      // Remove everything but the name
      $image = str_replace( ['texture-', '.png'], '', $image);
      // Add to textures array    
      $textures[] = $image;
    }
  }
}